package gov.va.med.mhv.usermgmt.service;

import junit.framework.TestCase;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.Message;
import org.tigris.atlas.messages.Messages;
import org.tigris.atlas.service.CollectionServiceResponse;
import org.tigris.atlas.service.EntityServiceResponse;
import org.tigris.atlas.service.ServiceResponse;
import org.tigris.atlas.transfer.TransferObject;

import gov.va.med.mhv.core.util.MessagesUtils;
import gov.va.med.mhv.core.util.ServiceResponseUtils;

/**
 * @author ckrahe
 */
public abstract class BaseTestCase extends TestCase {
    
    private static final Log LOG = LogFactory.getLog(BaseTestCase.class);
	
	public BaseTestCase() {
		super();
	}

	public BaseTestCase(String name) {
		super(name);
	}

    protected static boolean hasErrors(ServiceResponse response) {
        return ServiceResponseUtils.hasErrorMessages(response);
    }
	protected static boolean hasErrors(EntityServiceResponse response) {
        return ServiceResponseUtils.hasErrorMessages(response);
	}
	protected static boolean hasErrors(CollectionServiceResponse response) {
        return ServiceResponseUtils.hasErrorMessages(response);
	}
    
    protected static String getErrors(ServiceResponse response) {
        if (response == null) {
            return "";
        }
        return appendErrors(new StringBuilder(), response.getMessages()).
            toString();
    }
	protected static String getErrors(EntityServiceResponse response) {
        if (response == null) {
            return "";
        }
        StringBuilder builder = new StringBuilder(); 
        appendErrors(builder, response.getMessages());
		if (response.getEntity() != null) {
            appendErrors(builder, response.getEntity().getAllMessages());
        }
		return builder.toString();
	}
    protected static String getErrors(CollectionServiceResponse response) {
        if (response == null) {
            return "";
        }
        StringBuilder builder = new StringBuilder(); 
        appendErrors(builder, response.getMessages());
        if (response.getItems() != null) {
            for(Object item: response.getItems()) {
                appendErrors(builder, ((TransferObject) item).getAllMessages());
            }
        }
        return builder.toString();
    }

    protected static void assertNoErrors(ServiceResponse response) {
        assertNoErrors(null, response);
    }
    protected static void assertNoErrors(CollectionServiceResponse response) {
        assertNoErrors(null, response);
    }
    protected static void assertNoErrors(EntityServiceResponse response) {
        assertNoErrors(null, response);
    }
	protected static void assertNoErrors(String description, 
		ServiceResponse response) 
	{
		assertMessages(response);
		description = (StringUtils.isBlank(description)) 
            ? "" : " " + description;
		assertFalse("No error messages expected, but got " + 
            getErrors(response) + description, 
            ServiceResponseUtils.hasErrorMessages(response));
	}
    protected static void assertNoErrors(String description, 
        EntityServiceResponse response) 
    {
        assertMessages(response);
        description = (StringUtils.isBlank(description)) 
            ? "" : " " + description;
        assertFalse("No error messages expected, but got " 
            + getErrors(response) + description, 
            ServiceResponseUtils.hasErrorMessages(response));
    }
    protected static void assertNoErrors(String description, 
        CollectionServiceResponse response) 
    {
        assertMessages(response);
        description = (StringUtils.isBlank(description)) 
            ? "" : " " + description;
        assertFalse("No error messages expected, but got " 
            + getErrors(response) + description, 
            ServiceResponseUtils.hasErrorMessages(response));
    }
	
	protected static void assertHasError(ServiceResponse response, String key) { 
		assertHasError(null, response, key);
	}
    protected static void assertHasError(EntityServiceResponse response, 
        String key) 
    { 
        assertHasError(null, response, key);
    }
    protected static void assertHasError(CollectionServiceResponse response, 
        String key) 
    { 
        assertHasError(null, response, key);
    }
	protected static void assertHasError(String description, 
		ServiceResponse response, String key) 
	{
		assertNotBlank("No error key provided", key);
		assertMessages(response);
		if (!StringUtils.isBlank(description)) {
			description = description + ". ";
		}
		assertTrue(description + "Expected error '" + key 
            + "', but not found in " + getErrors(response), 
            ServiceResponseUtils.hasErrorMessage(response, key));
	}
    protected static void assertHasError(String description, 
        EntityServiceResponse response, String key) 
    {
        assertNotBlank("No error key provided", key);
        assertMessages(response);
        if (!StringUtils.isBlank(description)) {
            description = description + ". ";
        }
        assertTrue(description + "Expected error '" + key 
            + "', but not found in " + getErrors(response), 
            ServiceResponseUtils.hasErrorMessage(response, key));
    }
    protected static void assertHasError(String description, 
        CollectionServiceResponse response, String key) 
    {
        assertNotBlank("No error key provided", key);
        assertMessages(response);
        if (!StringUtils.isBlank(description)) {
            description = description + ". ";
        }
        assertTrue(description + "Expected error '" + key 
            + "', but not found in " + getErrors(response), 
            ServiceResponseUtils.hasErrorMessage(response, key));
    }

	protected static void assertNotHasError(ServiceResponse response, 
		String key) 
	{ 
		assertNotHasError(null, response, key);
	}
	protected static void assertNotHasError(String description, 
		ServiceResponse response, String key) 
	{
		assertNotBlank("No error key provided", key);
		assertMessages(response);
		if (StringUtils.isBlank(description)) {
			description = " " + description;
		}
		assertFalse("Expected Messages not to contain error with key '" + key 
			+ "'", MessagesUtils.hasErrorMessage(response.getMessages(), key));
	}

	protected static void assertNotBlank(String description, String value) {
		assertFalse("Expected non-blank value. " + description, StringUtils.
			isBlank(value));
	}

    protected static void logMessages(ServiceResponse response, Class clazz, 
        Log log) 
    {
        ServiceResponseUtils.logMessages(response, 
            (clazz != null) ? clazz : BaseTestCase.class,
            (log != null) ? log : LOG);
    }
    protected static void logMessages(EntityServiceResponse response, 
        Class clazz, Log log) 
    {
        ServiceResponseUtils.logMessages(response, 
            (clazz != null) ? clazz : BaseTestCase.class,
            (log != null) ? log : LOG);
    }
    protected static void logMessages(CollectionServiceResponse response, 
        Class clazz, Log log) 
    {
        ServiceResponseUtils.logMessages(response, 
            (clazz != null) ? clazz : BaseTestCase.class,
            (log != null) ? log : LOG);
    }

    private static void assertMessages(ServiceResponse response) {
        assertNotNull("Response expected", response);
        Messages messages = response.getMessages();
        assertNotNull("Messages missing", messages);
    }

    private static StringBuilder appendErrors(StringBuilder builder, 
        Messages messages) 
    {
        assertNotNull(builder);
        if ((messages != null) && (messages.getErrorMessages() != null)) {
            for (Object m: messages.getErrorMessages()) {
                builder.append(((Message) m).toString());
                builder.append("\n");
            }
        }
        return builder;
    }

    protected Log getLocalLog() {
        return LOG;
    }
}

